package cn.iinti.majora3.adr

import android.annotation.SuppressLint
import android.content.Context
import android.provider.Settings
import android.util.Log
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.content.edit
import cn.iinti.majora3.sdk.MajoraLogger
import cn.iinti.majora3.sdk.client.CtrManager
import cn.iinti.majora3.sdk.client.MajoraClient
import com.topjohnwu.superuser.Shell
import com.topjohnwu.superuser.ShellUtils
import org.apache.commons.lang3.BooleanUtils
import java.lang.Thread.sleep
import java.util.function.Supplier

var currentMajoraClient: MajoraClient? = null

const val SP_NAME = "config"


class ConfigItem(val configKey: String, defaultValue: String) {
    var configValue by mutableStateOf(
        majora3App.getSharedPreferences(SP_NAME, Context.MODE_PRIVATE)
            .getString(configKey, defaultValue)
            ?: defaultValue
    )

    fun updateConfigValue(value: String) {
        majora3App.getSharedPreferences(SP_NAME, Context.MODE_PRIVATE)
            .edit(commit = true) {
                apply {
                    this.putString(configKey, value)
                }
            }
        configValue = value

        ensureMajorClient()
    }

}

var serverEndpoint = ConfigItem("serverEndpoint", "majora3.iinti.cn:6879")
var clientGroup = ConfigItem("clientGroup", "android")
var extra = ConfigItem("extra", "")

// todo 需要在页面上显示是否支持重播
var supportRedial = Shell.isAppGrantedRoot()


@Synchronized
fun ensureMajorClient() {
    if (majora3App.hasNetwork.value == false) {
        // 现在还没有网络
        return
    }

    val config = serverEndpoint.configValue.trim().split(':')
    if (config.size != 2) {
        uiLog("服务器配置不合理：" + serverEndpoint.configValue)
        return
    }
    val serverHost = config[0].trim()
    val serverPort = try {
        config[1].trim().toInt()
    } catch (t: Throwable) {
        uiLog("端口错误:" + serverEndpoint.configValue, t)
        return
    }

    if (currentMajoraClient == null) {
        currentMajoraClient = doCreateMajoraClient(serverHost, serverPort)
        return
    }

    if (currentMajoraClient!!.matchConfig(
            serverHost, serverPort,
            clientGroup.configValue, extra.configValue
        )
    ) {
        return
    }
    currentMajoraClient!!.destroy()
    currentMajoraClient = doCreateMajoraClient(serverHost, serverPort)
}

fun performRedial(client: MajoraClient) { 
    client.logger.apply { 
    // 启动飞行模式，断网
    sleep(200) 
    log { "开始重播，打开飞行模式 " } 
    val enableResult = ShellUtils.fastCmd("sh", "-c", "/system/bin/cmd connectivity airplane-mode enable").trim()
    log { "[enableResult] ${if (enableResult.isEmpty()) "(无输出)" else enableResult}" }
    log { "休眠7s，等待飞行模式打开完成" }
    sleep(7_000L)
    log { "关闭飞行模式" }
    val disableResult = ShellUtils.fastCmd("sh", "-c", "/system/bin/cmd connectivity airplane-mode disable").trim()
    log { "[disableResult] ${if (disableResult.isEmpty()) "(无输出)" else disableResult}" }
    log { "飞行模式切换完成，等待网络重新连接" } 
} }


class LogRecord(val msg: String, val throwable: Throwable?) {}

class LogBuffer(private val capacity: Int) {
    private val buffer = arrayOfNulls<LogRecord>(capacity)
    private var currentIndex = 0
    private var isFull = false

    fun add(msg: String, throwable: Throwable? = null) {
        buffer[currentIndex] = LogRecord(msg, throwable)
        currentIndex = (currentIndex + 1) % capacity
        if (currentIndex == 0) isFull = true
    }

    fun takeAll(): List<LogRecord> {
        return if (isFull) {
            buffer.toList().subList(currentIndex, capacity) + buffer.toList()
                .subList(0, currentIndex)
        } else {
            buffer.toList().take(currentIndex)
        }.filterNotNull().apply {
            currentIndex = 0
            isFull = false
        }
    }
}

val logBuffer = LogBuffer(10)

interface UILogListener {
    fun onLogMessage(message: String)
}

fun UILogListener.uiLog(msg: String, throwable: Throwable? = null) {
    onLogMessage(msg)
    if (throwable != null) {
        onLogMessage(throwable.message ?: "Exception occurred")
        onLogMessage(throwable.stackTraceToString())
    }
}

private var uILogListener: UILogListener? = null

fun registerUILogListener(logListener: UILogListener) {
    logBuffer.takeAll().forEach {
        logListener.uiLog(it.msg, it.throwable)
    }
    uILogListener = logListener
}

fun clearUILogListener() {
    uILogListener = null
}

private fun uiLog(message: String, e: Throwable? = null) {
    uILogListener.apply {
        if (this == null) {
            logBuffer.add(message, e)
        } else {
            uiLog(message, e)
        }
    }
}


@SuppressLint("HardwareIds")
fun doCreateMajoraClient(serverHost: String, serverPort: Int): MajoraClient {
    val clientId = Settings.Secure.getString(majora3App.contentResolver, Settings.Secure.ANDROID_ID)

    // todo 支持 pty
    val client = MajoraClient(serverHost, serverPort, clientId)
    client.endpointGroup = clientGroup.configValue
    client.extra = extra.configValue

    if (BooleanUtils.isTrue(Shell.isAppGrantedRoot())) {
        // 只有当app拥有root权限时，当前设备才支持重播，这和majoraV2不一样
        // majoraV2支持通过adb权限实现重播，然而adb权限非常不稳定，所以V3时代已经移除了相关支持
        CtrManager.registerCtrHandler(CtrManager.DEFAULT_ACTION_REDIAL) { request, response ->
            response.success(mapOf("msg" to "ok"))

            Thread {
                performRedial(client)
            }.start()
        }
    }

    client.logger = object : MajoraLogger {
        override fun log(msg: Supplier<String>, e: Throwable?) {
            val message = msg.get()
            if (e == null) {
                Log.i(TAG, message)
            } else {
                Log.i(TAG, message, e)
            }

            uiLog(message, e)
        }

        override fun log(msg: Supplier<String>) {
            this.log(msg, null)
        }
    }

    client.start()
    return client
}